<?php
// instrumentos_bulk_save.php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

// MIENTRAS DEPURAS, djalo activo; luego puedes quitarlo
ini_set('display_errors', '1');
error_reporting(E_ALL);

require_once __DIR__ . '/includes/db.php';

function out_ok($data = []) {
  echo json_encode(array_merge(['ok' => true], $data));
  exit;
}
function out_err($msg, $http_code = 400) {
  http_response_code($http_code);
  echo json_encode(['ok' => false, 'error' => $msg]);
  exit;
}

try {
  // ---------- Entrada ----------
  $raw = file_get_contents('php://input');
  $payload = json_decode($raw, true);
  if (!is_array($payload)) {
    // fallback si te llega como form-urlencoded
    $payload = $_POST ?: [];
  }

  $id_modulo   = isset($payload['id_modulo']) ? (int)$payload['id_modulo'] : 0;
  $anno        = isset($payload['anno']) ? trim((string)$payload['anno']) : '';
  $titulo      = isset($payload['titulo']) ? trim((string)$payload['titulo']) : '';
  $descripcion = isset($payload['descripcion']) && $payload['descripcion'] !== '' ? (string)$payload['descripcion'] : null;
  $fecha       = isset($payload['fecha']) && $payload['fecha'] !== '' ? (string)$payload['fecha'] : null;
  $trimestre   = isset($payload['trimestre']) ? (int)$payload['trimestre'] : null;
  $ponderacion = (isset($payload['ponderacion']) && $payload['ponderacion'] !== '') ? (int)$payload['ponderacion'] : null;

  $criterios     = (isset($payload['criterios']) && is_array($payload['criterios'])) ? array_map('intval', $payload['criterios']) : [];
  $asignar_todos = !empty($payload['asignar_todos']);
  $alumnosSel    = (isset($payload['alumnos']) && is_array($payload['alumnos'])) ? array_map('intval', $payload['alumnos']) : [];

  if ($id_modulo <= 0) out_err("Falta id_modulo.");
  if ($anno === '')     out_err("Falta anno (p.ej. 2025/2026).");
  if ($titulo === '')   out_err("Falta titulo.");
  if (!in_array($trimestre, [1,2,3], true)) out_err("Trimestre invlido (1,2,3).");

  // ---------- Conexin ----------
  $pdo = pdo_conn();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  // ---------- Transaccin ----------
  $pdo->beginTransaction();

  // ---------- Insert instrumento ----------
  // Requiere columna instrumentos.id_modulo; si falla, devolvemos instruccin SQL sugerida
  $sqlIns = "
    INSERT INTO instrumentos
      (id_modulo, anno, trimestre, titulo, descripcion, fecha, ponderacion, creado_at)
    VALUES
      (:id_modulo, :anno, :trimestre, :titulo, :descripcion, :fecha, :ponderacion, NOW())
  ";
  try {
    $stIns = $pdo->prepare($sqlIns);
    $stIns->execute([
      ':id_modulo'   => $id_modulo,
      ':anno'        => $anno,
      ':trimestre'   => $trimestre,
      ':titulo'      => $titulo,
      ':descripcion' => $descripcion,
      ':fecha'       => $fecha,
      ':ponderacion' => $ponderacion,
    ]);
  } catch (PDOException $e) {
    if ($e->getCode() === '42S22' && stripos($e->getMessage(), 'id_modulo') !== false) {
      $pdo->rollBack();
      out_err("Tu tabla 'instrumentos' no tiene la columna 'id_modulo'. Aplica esta migracin:\n\n".
        "ALTER TABLE instrumentos\n".
        "  ADD COLUMN id_modulo INT NOT NULL AFTER id,\n".
        "  ADD INDEX idx_instrumentos_modulo (id_modulo);\n", 500);
    }
    $pdo->rollBack();
    out_err("Error al insertar en 'instrumentos': ".$e->getMessage(), 500);
  }

  $id_instrumento = (int)$pdo->lastInsertId();

  // ---------- Vincular criterios seleccionados ----------
  if (!empty($criterios)) {
    $sqlC = "
      INSERT INTO instrumentos_criterios (id_instrumento, id_criterio_evaluacion)
      VALUES (:id_instrumento, :id_criterio)
    ";
    try {
      $stC = $pdo->prepare($sqlC);
      foreach ($criterios as $idCrit) {
        if ($idCrit > 0) {
          $stC->execute([
            ':id_instrumento' => $id_instrumento,
            ':id_criterio'    => $idCrit,
          ]);
        }
      }
    } catch (PDOException $e) {
      $pdo->rollBack();
      out_err("Error al vincular criterios con el instrumento: ".$e->getMessage(), 500);
    }
  }

  // ---------- Obtener alumnos destinatarios ----------
  $idsAlumnos = [];

  if ($asignar_todos) {
    // Detectar si alumnos_modulos tiene columna 'anno'
    $cols = $pdo->query("SHOW COLUMNS FROM alumnos_modulos")->fetchAll(PDO::FETCH_COLUMN, 0);
    $hasAnno = false;
    if ($cols) {
      foreach ($cols as $c) {
        if ($c === 'anno') { $hasAnno = true; break; }
      }
    }

    if ($hasAnno) {
      $sqlMat = "
        SELECT id_alumno
        FROM alumnos_modulos
        WHERE id_modulo = :id_modulo AND anno = :anno
      ";
      $q = $pdo->prepare($sqlMat);
      $q->execute([':id_modulo' => $id_modulo, ':anno' => $anno]);
    } else {
      // Sin anno en alumnos_modulos: filtramos solo por mdulo
      $sqlMat = "
        SELECT id_alumno
        FROM alumnos_modulos
        WHERE id_modulo = :id_modulo
      ";
      $q = $pdo->prepare($sqlMat);
      $q->execute([':id_modulo' => $id_modulo]);
    }

    $rows = $q->fetchAll(PDO::FETCH_ASSOC);
    $idsAlumnos = [];
    foreach ($rows as $r) {
      $idsAlumnos[] = (int)$r['id_alumno'];
    }
  } else {
    $idsAlumnos = $alumnosSel;
  }

  // ---------- Crear filas en instrumentos_notas (nota NULL) ----------
  if (!empty($idsAlumnos)) {
    $sqlN = "
      INSERT IGNORE INTO instrumentos_notas (id_instrumento, id_alumno, nota, fecha)
      VALUES (:id_instrumento, :id_alumno, NULL, NOW())
    ";
    try {
      $stN = $pdo->prepare($sqlN);
      foreach ($idsAlumnos as $idAl) {
        if ($idAl > 0) {
          $stN->execute([
            ':id_instrumento' => $id_instrumento,
            ':id_alumno'      => $idAl,
          ]);
        }
      }
    } catch (PDOException $e) {
      $pdo->rollBack();
      out_err("Error al crear las filas de notas para alumnos: ".$e->getMessage(), 500);
    }
  }

  // ---------- Commit y OK ----------
  $pdo->commit();
  out_ok(['id_instrumento' => $id_instrumento]);

} catch (Throwable $e) {
  if (isset($pdo) && $pdo->inTransaction()) {
    $pdo->rollBack();
  }
  out_err("Excepcin no controlada: ".$e->getMessage(), 500);
}
